#include "cybsp.h"
#include "cyhal.h"

#include "LEDS.h"
#include "SPILED.h"
#include "MAX40080.h"
#include "UART.h"

#include <string.h>
#include <stdio.h>

#define CSA_EN_PIN P9_7

bool csaInitialized = false;

void InitCsa() {
	MAX40080_Status mStatus;

	cyhal_gpio_write(CSA_EN_PIN, 0);
	cyhal_system_delay_ms(50);
	cyhal_gpio_write(CSA_EN_PIN, 1);
	cyhal_system_delay_ms(50);

	mStatus = MAX40080_Init();
	if (mStatus != MAX40080_Status_Ok) {
		return;
	}

	MAX40080_Configuration cfg;
	mStatus = MAX40080_GetDefaultConfiguration(&cfg);
	if (mStatus != MAX40080_Status_Ok) {
		return;
	}

	cfg.operatingMode = MAX40080_OperationMode_Active;
	cfg.packetErrorChecking = MAX40080_PacketErrorChecking_Enabled;
	cfg.inputRange = MAX40080_InputRange_10mV;
	cfg.adcSampleRate = MAX40080_AdcSampleRate_Either_at_15_ksps;
	cfg.digitalFilter = MAX40080_DigitalFilter_32_samples;

	mStatus = MAX40080_SetConfiguration(&cfg);
	if (mStatus != MAX40080_Status_Ok) {
		return;
	}

	csaInitialized = true;
}

int ReadCsa(int16_t* measuredCurrent) {
	MAX40080_Status mStatus;

	int attemps = 10;
	while (attemps-- > 0) {
		if (!csaInitialized) {
			InitCsa();
		}
		if (!csaInitialized) {
			cyhal_system_delay_ms(100);
			continue;
		}

		mStatus = MAX40080_FlushFifo();
		if (mStatus != MAX40080_Status_Ok) {
			csaInitialized = false;
			continue;
		}

		cyhal_system_delay_ms(1);

		int16_t current;
		mStatus = MAX40080_ReadRawCurrent(&current);
		if (mStatus == MAX40080_Status_FifoIsEmpty) {
			cyhal_system_delay_ms(1);
			mStatus = MAX40080_ReadRawCurrent(&current);
		}
		if (mStatus == MAX40080_Status_Ok) {
			*measuredCurrent = current;
			return 0;
		} else {
			csaInitialized = false;
			continue;
		}
	}

	return 1;
}

void AnalyzeChain();

int main(void) {
	cy_rslt_t result;

	result = cybsp_init();
	CY_ASSERT(result == CY_RSLT_SUCCESS);

	__enable_irq();
	cyhal_system_delay_ms(100);

	LEDS_Init();
	SPILED_Init();
	UART_Init();

	UART_SetFontColor(UART_COLOR_WHITE);
	UART_PrintString("\r\nInit done\r\n");

	cyhal_gpio_init(CYBSP_USER_BTN2, CYHAL_GPIO_DIR_INPUT, CYHAL_GPIO_DRIVE_NONE, 0);

	while (1) {
		while (cyhal_gpio_read(CYBSP_USER_BTN2) == 1) {}
		while (cyhal_gpio_read(CYBSP_USER_BTN2) == 0) {}

		AnalyzeChain();
	}
}

typedef struct {
	int16_t idleCurrent;
	size_t physicalLedsCount;
	size_t scanIndex;
	size_t badCount;
	int isDone;
	int isFailed;
} AnalysisContext;

#define UPDATE_AND_READ_CURRENT() LEDS_Transmit(); cyhal_system_delay_ms(150); status = ReadCsa(&current); if (status) { ctx->isFailed = 1; return; }

#define CURRENT_INCREASE_THRESHOLD_24BIT_PWM 20
#define CURRENT_INCREASE_THRESHOLD_24BIT_PWM_MIDRANGE 10
#define CURRENT_INCREASE_THRESHOLD_48BIT_PWM 6
#define CURRENT_INCREASE_THRESHOLD_48BIT_PWM_MIDRANGE 3
#define CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT 10
#define CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT_MIDRANGE 5

void Analyze24BitLed(AnalysisContext* ctx) {
	int status;
	int16_t current;

#define TEST(name, r, g, b) \
		LEDS_SetLed24(ctx->scanIndex, r, g, b); \
		UPDATE_AND_READ_CURRENT(); \
		int16_t name = current - ctx->idleCurrent;

	TEST(redHalf, 128, 0, 0);
	TEST(redFull, 255, 0, 0);
	TEST(greenHalf, 0, 128, 0);
	TEST(greenFull, 0, 255, 0);
	TEST(blueHalf, 0, 0, 128);
	TEST(blueFull, 0, 0, 255);

#undef TEST

	LEDS_SetLed24(ctx->scanIndex, 0, 0, 0);

	bool redOk =
		((redHalf - ctx->idleCurrent) > CURRENT_INCREASE_THRESHOLD_24BIT_PWM_MIDRANGE) &&
		((redFull - redHalf) > CURRENT_INCREASE_THRESHOLD_24BIT_PWM_MIDRANGE);

	bool greenOk =
		((greenHalf - ctx->idleCurrent) > CURRENT_INCREASE_THRESHOLD_24BIT_PWM_MIDRANGE) &&
		((greenFull - greenHalf) > CURRENT_INCREASE_THRESHOLD_24BIT_PWM_MIDRANGE);

	bool blueOk =
		((blueHalf - ctx->idleCurrent) > CURRENT_INCREASE_THRESHOLD_24BIT_PWM_MIDRANGE) &&
		((blueFull - blueHalf) > CURRENT_INCREASE_THRESHOLD_24BIT_PWM_MIDRANGE);

	UART_SetFontColor(UART_COLOR_YELLOW);
	UART_PrintString("1W ");
	UART_SetFontColor(UART_COLOR_CYAN);
	UART_PrintString("24-BIT\t");
	if (redOk) {
		UART_SetFontColor(UART_COLOR_GREEN);
		UART_PrintString("R=OK  ");
	} else {
		UART_SetFontColor(UART_COLOR_RED);
		UART_PrintString("R=BAD ");
	}
	if (greenOk) {
		UART_SetFontColor(UART_COLOR_GREEN);
		UART_PrintString("G=OK  ");
	} else {
		UART_SetFontColor(UART_COLOR_RED);
		UART_PrintString("G=BAD ");
	}
	if (blueOk) {
		UART_SetFontColor(UART_COLOR_GREEN);
		UART_PrintString("B=OK  ");
	} else {
		UART_SetFontColor(UART_COLOR_RED);
		UART_PrintString("B=BAD ");
	}
	UART_PrintString("\r\n");
}

void Analyze48BitLed(AnalysisContext* ctx) {
	int status;
	int16_t current;

#define TEST(name, iR, pR, iG, rG, iB, pB) \
		LEDS_SetLed48(ctx->scanIndex, iR, pR, iG, rG, iB, pB); \
		UPDATE_AND_READ_CURRENT(); \
		volatile int16_t name = current - ctx->idleCurrent;

	TEST(redHalfPwm, 0, 2048, 0, 0, 0, 0);
	TEST(redFullPwm, 0, 4095, 0, 0, 0, 0);
	TEST(greenHalfPwm, 0, 0, 0, 2048, 0, 0);
	TEST(greenFullPwm, 0, 0, 0, 4095, 0, 0);
	TEST(blueHalfPwm, 0, 0, 0, 0, 0, 2048);
	TEST(blueFullPwm, 0, 0, 0, 0, 0, 4095);
	TEST(redHalfCurrent, 8, 4095, 0, 0, 0, 0);
	TEST(redFullCurrent, 15, 4095, 0, 0, 0, 0);
	TEST(greenHalfCurrent, 0, 0, 3, 4095, 0, 0);
	TEST(greenFullCurrent, 0, 0, 15, 4095, 0, 0);
	TEST(blueHalfCurrent, 0, 0, 0, 0, 3, 4095);
	TEST(blueFullCurrent, 0, 0, 0, 0, 15, 4095);

#undef TEST

	LEDS_SetLed48(ctx->scanIndex, 0, 0, 0, 0, 0, 0);

	bool redPwmOk =
		(redHalfPwm > CURRENT_INCREASE_THRESHOLD_48BIT_PWM_MIDRANGE) &&
		((redFullPwm - redHalfPwm) > CURRENT_INCREASE_THRESHOLD_48BIT_PWM_MIDRANGE);

	bool greenPwmOk =
		(greenHalfPwm > CURRENT_INCREASE_THRESHOLD_48BIT_PWM_MIDRANGE) &&
		((greenFullPwm - greenHalfPwm) > CURRENT_INCREASE_THRESHOLD_48BIT_PWM_MIDRANGE);

	bool bluePwmOk =
		(blueHalfPwm > CURRENT_INCREASE_THRESHOLD_48BIT_PWM_MIDRANGE) &&
		((blueFullPwm - blueHalfPwm) > CURRENT_INCREASE_THRESHOLD_48BIT_PWM_MIDRANGE);

	bool redCurrentOk =
		(redHalfCurrent > CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT_MIDRANGE) &&
		((redFullCurrent - redHalfCurrent) > CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT_MIDRANGE);

	bool greenCurrentOk =
		(greenHalfCurrent > CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT_MIDRANGE) &&
		((greenFullCurrent - greenHalfCurrent) > CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT_MIDRANGE);

	bool blueCurrentOk =
		(blueHalfCurrent > CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT_MIDRANGE) &&
		((blueFullCurrent - blueHalfCurrent) > CURRENT_INCREASE_THRESHOLD_48BIT_CURRENT_MIDRANGE);

	bool redOk = redPwmOk && redCurrentOk;
	bool greenOk = greenPwmOk && greenCurrentOk;
	bool blueOk = bluePwmOk && blueCurrentOk;

	UART_SetFontColor(UART_COLOR_YELLOW);
	UART_PrintString("1W ");
	UART_SetFontColor(UART_COLOR_MAGENTA);
	UART_PrintString("48-BIT\t");
	if (redOk) {
		UART_SetFontColor(UART_COLOR_GREEN);
		UART_PrintString("R=OK  ");
	} else {
		UART_SetFontColor(UART_COLOR_RED);
		UART_PrintString("R=BAD ");
	}
	if (greenOk) {
		UART_SetFontColor(UART_COLOR_GREEN);
		UART_PrintString("G=OK  ");
	} else {
		UART_SetFontColor(UART_COLOR_RED);
		UART_PrintString("G=BAD ");
	}
	if (blueOk) {
		UART_SetFontColor(UART_COLOR_GREEN);
		UART_PrintString("B=OK  ");
	} else {
		UART_SetFontColor(UART_COLOR_RED);
		UART_PrintString("B=BAD ");
	}
	UART_PrintString("\r\n");
}

void AnalyzeNextLed(AnalysisContext* ctx) {
	int status;
	int16_t current;


	if (ctx->scanIndex >= LEDS_MAX_LEDS) {
		UART_PrintString("Chain is longer than 4096 LEDs\r\n");
		return;
	}

	// identify if it is 24-bit or 48-bit
	// 0xF00000000000 is no color on 48-bit because Ired=full, but PWM=0
	// 0xF00000       is quite bright red on 24-bit.
	// teh same we will try for red and blue.
	LEDS_SetLed48(ctx->scanIndex, 15, 0, 0, 0, 0, 0);
	UPDATE_AND_READ_CURRENT();
	int16_t red48OffIncrease = current - ctx->idleCurrent;

	LEDS_SetLed48(ctx->scanIndex, 0, 0, 15, 0, 0, 0);
	UPDATE_AND_READ_CURRENT();
	int16_t green48OffIncrease = current - ctx->idleCurrent;

	LEDS_SetLed48(ctx->scanIndex, 0, 0, 0, 0, 15, 0);
	UPDATE_AND_READ_CURRENT();
	int16_t blue48OffIncrease = current - ctx->idleCurrent;

	LEDS_SetLed48(ctx->scanIndex, 6, 2048, 6, 2048, 6, 2048);
	UPDATE_AND_READ_CURRENT();
	int16_t on48Increate = current - ctx->idleCurrent;

	LEDS_SetLed48(ctx->scanIndex, 0, 0, 0, 0, 0, 0);
	LEDS_SetLed48(ctx->scanIndex + 2, 6, 2048, 6, 2048, 6, 2048);
	UPDATE_AND_READ_CURRENT();
	int16_t nextOn48Increate = current - ctx->idleCurrent;

	LEDS_SetLed48(ctx->scanIndex + 2, 0, 0, 0, 0, 0, 0);

	if (red48OffIncrease > CURRENT_INCREASE_THRESHOLD_48BIT_PWM ||
		green48OffIncrease > CURRENT_INCREASE_THRESHOLD_48BIT_PWM ||
		blue48OffIncrease > CURRENT_INCREASE_THRESHOLD_48BIT_PWM) {
		Analyze24BitLed(ctx);
		ctx->physicalLedsCount++;
		ctx->scanIndex += 1;
	} else if (on48Increate > CURRENT_INCREASE_THRESHOLD_48BIT_PWM) {
		Analyze48BitLed(ctx);
		ctx->physicalLedsCount++;
		ctx->scanIndex += 2;
	} else if (nextOn48Increate > CURRENT_INCREASE_THRESHOLD_48BIT_PWM) {
		Analyze48BitLed(ctx);
		ctx->physicalLedsCount++;
		ctx->scanIndex += 2;
		ctx->badCount++;
	} else {
		UART_SetFontColor(UART_COLOR_WHITE);
		UART_PrintString("END OF STRIP\r\n");
		ctx->isDone = 1;
	}
}

void AnalyzeChain() {
	UART_SetFontColor(UART_COLOR_WHITE);
	UART_PrintString("=====================================\r\n");
	UART_PrintString("==    Starting analysis of chain   ==\r\n");
	UART_PrintString("=====================================\r\n");

	// clear buffer
	for (size_t i = 0; i < LEDS_MAX_LEDS; i++) {
		LEDS_SetLed24(i, 0, 0, 0);
	}
	LEDS_Transmit();
	cyhal_system_delay_ms(150);

	AnalysisContext ctx;
	ctx.badCount = 0;
	ctx.physicalLedsCount = 0;
	ctx.scanIndex = 0;
	ctx.isDone = 0;
	ctx.isFailed = 0;
	ReadCsa(&ctx.idleCurrent);

	while (!ctx.isDone && !ctx.isFailed) {
		char buff[32];
		snprintf(buff, sizeof(buff), "LED %4d ", ctx.physicalLedsCount);
		UART_SetFontColor(UART_COLOR_WHITE);
		UART_PrintString(buff);

		AnalyzeNextLed(&ctx);
	}

	if (ctx.isFailed) {
		UART_SetFontColor(UART_COLOR_RED);
		UART_PrintString("Strip analysis failed due to error.\r\n");
	}

	UART_SetFontColor(UART_COLOR_WHITE);
	UART_PrintString("=====================================\r\n");
}
