#include "LEDS.h"
#include "cy_scb_spi.h"
#include "cy_dma.h"

static uint8_t buffer[LEDS_RESET_BYTES + LEDS_MAX_LEDS * 12];

static cy_stc_scb_spi_context_t spiCtx;
static cy_stc_dma_descriptor_t dmaDescriptor;


void LEDS_Init() {
	for (size_t i = 0; i < LEDS_RESET_BYTES; i++) {
        buffer[i] = 0;
    }
	for (size_t i = 0; i < LEDS_MAX_LEDS * 12; i++) {
		buffer[LEDS_RESET_BYTES + i] = 0x88;
	}

	cy_rslt_t status;

    NVIC_ClearPendingIRQ(scb_1_interrupt_IRQn);
    NVIC_EnableIRQ(scb_1_interrupt_IRQn);

	cy_stc_scb_spi_config_t spiConfig;
    spiConfig.spiMode = CY_SCB_SPI_MASTER;
    spiConfig.subMode = CY_SCB_SPI_MOTOROLA;
    spiConfig.sclkMode = CY_SCB_SPI_CPHA0_CPOL0;
    spiConfig.oversample = 4;
    spiConfig.rxDataWidth = 8UL;
    spiConfig.txDataWidth = 8UL;
    spiConfig.enableMsbFirst = true;
    spiConfig.enableInputFilter = false;
    spiConfig.enableFreeRunSclk = false;
    spiConfig.enableMisoLateSample = true;
    spiConfig.enableTransferSeperation = false;
    spiConfig.ssPolarity = 0;
    spiConfig.enableWakeFromSleep = false;
    spiConfig.rxFifoTriggerLevel = 63UL;
    spiConfig.rxFifoIntEnableMask = 0UL;
    spiConfig.txFifoTriggerLevel = 63UL;
    spiConfig.txFifoIntEnableMask = 0UL;
    spiConfig.masterSlaveIntEnableMask = 0UL;

    status = Cy_SCB_SPI_Init(SCB1, &spiConfig, &spiCtx);
    CY_ASSERT(status == CY_RSLT_SUCCESS);
    Cy_SCB_SPI_Enable(SCB1);

	cy_stc_dma_descriptor_config_t descriptorConfig;
    descriptorConfig.channelState = CY_DMA_CHANNEL_DISABLED;
    descriptorConfig.interruptType = CY_DMA_DESCR;
    descriptorConfig.triggerOutType = CY_DMA_DESCR;
    descriptorConfig.triggerInType = CY_DMA_1ELEMENT;
    descriptorConfig.srcXincrement = 1;
    descriptorConfig.dstXincrement = 0;
    descriptorConfig.xCount = 256;
    descriptorConfig.srcYincrement = 256;
    descriptorConfig.dstYincrement = 0;
    descriptorConfig.yCount = sizeof(buffer) / 256;
    descriptorConfig.descriptorType = CY_DMA_2D_TRANSFER;
    descriptorConfig.dataSize = CY_DMA_BYTE;
    descriptorConfig.dstAddress = (void *)&(SCB1->TX_FIFO_WR);
	descriptorConfig.retrigger = CY_DMA_RETRIG_IM;
    descriptorConfig.srcTransferSize = CY_DMA_TRANSFER_SIZE_DATA;
    descriptorConfig.dstTransferSize = CY_DMA_TRANSFER_SIZE_WORD;
    descriptorConfig.srcAddress = buffer;
    descriptorConfig.nextDescriptor = NULL;

    status = Cy_DMA_Descriptor_Init(&dmaDescriptor, &descriptorConfig);
    CY_ASSERT(status == 0);

    NVIC_SetPriority(cpuss_interrupts_dw0_18_IRQn, 3);
    NVIC_ClearPendingIRQ(cpuss_interrupts_dw0_18_IRQn);
    NVIC_EnableIRQ(cpuss_interrupts_dw0_18_IRQn);

	Cy_DMA_Channel_SetDescriptor(DW0, 18, &dmaDescriptor);
    Cy_DMA_Channel_SetPriority(DW0, 18, 2);
    Cy_DMA_Channel_SetInterruptMask(DW0, 18, CY_DMA_INTR_MASK);
}

static void LEDS_SetByte(uint8_t* destination, uint8_t value) {
	for (int i = 6; i >= 0; i -= 2) {
        if (value & (1 << (i + 1))) {
            *destination |= 0x60;
        } else {
            *destination &= ~0x60;
        }
        if (value & (1 << (i + 0))) {
            *destination |= 0x06;
        } else {
            *destination &= ~0x06;
        }
        *destination++;
	}
}

void LEDS_SetLed24(size_t index, uint8_t r, uint8_t g, uint8_t b) {
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 0, g);
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 4, r);
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 8, b);
}

void LEDS_SetLed48(size_t index, uint8_t rI, uint16_t rPWM, uint8_t gI, uint16_t gPWM, uint8_t bI, uint16_t bPWM) {
    CY_ASSERT(rI < 16);
    CY_ASSERT(rPWM < 4096);
    CY_ASSERT(gI < 16);
    CY_ASSERT(gPWM < 4096);
    CY_ASSERT(bI < 16);
    CY_ASSERT(bPWM < 4096);

	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 0, (rI << 4) | (rPWM >> 8));
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 4, rPWM);
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 8, (gI << 4) | (gPWM >> 8));
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 12, gPWM);
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 16, (bI << 4) | (bPWM >> 8));
	LEDS_SetByte(buffer + LEDS_RESET_BYTES + index * 12 + 20, bPWM);
}


void scb_1_interrupt_IRQHandler(void) {
    Cy_SCB_SPI_Interrupt(SCB1, &spiCtx);
}

void cpuss_interrupts_dw0_18_IRQHandler(void) {
    cy_en_dma_intr_cause_t status = Cy_DMA_Channel_GetStatus(DW0, 18);
    if (status != CY_DMA_INTR_CAUSE_COMPLETION) {
        __BKPT(0);
    }
    Cy_DMA_Channel_Disable(DW0, 18);
    Cy_DMA_Channel_ClearInterrupt(DW0, 18);
}

void LEDS_Transmit() {
	Cy_DMA_Channel_SetDescriptor(DW0, 18, &dmaDescriptor);
    Cy_DMA_Enable(DW0);
    Cy_DMA_Channel_Enable(DW0, 18);
}